// ExperimenterPlugin.cpp starts

/* Created: CGP, 12/21/00
 *		Common code for experimenters.
 * Modified, CGP, 1/5/01
 *		Experimenter plugins now have two threads. The extra thread
 *		is to receive incoming messages on the port and enqueue them.
 *		This will make sure no threads sending us messages block when
 *		they are sending us the message.
 * Modified: CGP, 1/6/01
 *		Setup virtual function called from message loop in the Start thread.
 *		This is so that the user will have access to the full experimenter
 *		API when coding the Setup function.
 */

#include <Message.h>
#include <OS.h>
#include <stdio.h>
#include "ExperimenterPlugin.h"
#include "DatabaseFieldNames.h"
#include "UserEnvMessage.h"
#include "Debug.h"
#include "EnvironmentConstants.h"
#include "WaitQueue.h"
#include "PluginSettingsNames.h"

// Globals for experimenters, but not for simulator
#ifndef MAZE_SIMULATOR
DebugServer *GLOB_debug = NULL;
#endif

ExperimenterPlugin::ExperimenterPlugin(PortMessage *expRatPort,
	PortMessage *envExpPort, DatabaseMessage *dbMsg, UserEnvMessage *userEnvMsg,
	DebugServer *bugServer)
{
	this->envExpPort = envExpPort;
	this->expRatPort = expRatPort;
	this->dbMsg = dbMsg;
	this->userEnvMsg = userEnvMsg;
	this->queue = new WaitQueue();
	GLOB_debug = bugServer;
}


ExperimenterPlugin::~ExperimenterPlugin()
{
	delete queue;
}

bool ExperimenterPlugin::VersionOK(){
	if ((userEnvMsg->Ver() == USER_ENV_MESSAGE_API_VER) &&
		(PLUGIN_API_Version() == EXPERIMENTER_API_VER)) {
		return true;
	} else {
		return false;
	}
}

// Parameter is NULL, as we don't need it.
// This is the function for the process to receive messages from the experimenter's port.
// Because this is declared as a friend, we can access the private members.
long ReceiveMessages(void *data) // data is the ExperimenterPlugin *
{
	char buf[50];
	
	Debug('n', "ExperimenterPlugin.ReceiveMessages: Started");
	ExperimenterPlugin *exp = (ExperimenterPlugin *) data;
	WaitQueue *queue = exp->queue;
	
	// Fake the first message into the queue, so we can have the virtual Setup function
	// called as the first experimenter plugin function called.
	BMessage *setup = new BMessage(EXPMSG_SETUP);
	queue->Add(setup);
	// Don't delete the message-- we give up ownership passing to Add.
	
	for (;;) {
		// block waiting for a port message from environment or from GUI
		BMessage *msg = exp->envExpPort->GetMessage();
		sprintf(buf, "%d", (int) msg->what);
		
		queue->Add(msg);
		// Don't delete the msg; we give up ownership passing it to Add.
	}
}

// Run the plugin. We start up two threads. One reads messages and stores them in
// a queue. The other consumes these messages. We need a separate process to read
// the messages into a queue so that processes sending to us don't block
// waiting for a message to be sent. (e.g., the environment sending a message to us
// updating us on the status of the rat).
long ExperimenterPlugin::Start(void)
{
	char buf[50];
	
	Debug('n', "ExperimenterPlugin::Start: Started");
	
	// Start a thread to receive messages. First, make the thread.
	receiveMessagesThreadID = spawn_thread(ReceiveMessages,
		"receive messages", B_NORMAL_PRIORITY, (void *) this);
	if (receiveMessagesThreadID < 0) {
		// failure
		Debug('e', "ExperimenterPlugin::Start: Error: Failed to start ReceiveMessage thread!");
    }
    
    // Now, start the thread.
    if (resume_thread(receiveMessagesThreadID) != B_OK) {
    	Debug('e', "ExperimenterPlugin::Start: Error: Could not resume_thread");
    }
	
	for (;;) {
		Debug('n', "ExperimenterPlugin::Start: Getting a message...");
		// grab the next incoming message from the port queue
		BMessage *msg = queue->Remove();
		
		// Call the virtual functions
		// (defined in current experimenter plugin or defaults in ExperimenterPlugin.h)
		switch (msg->what) {
		case EXPMSG_SETUP:
			PLUGIN_Setup(); // Will only be called once.
			break;
		
		case EXPMSG_RUN_TRIAL:
			PLUGIN_RunTrial();
			break;
			
		case EXPMSG_COMPLETE_CURRENT_RAT:
			PLUGIN_RunCurrentRat();
			break;
			
		case EXPMSG_RAT_MOVED:
			PLUGIN_RatMoved(msg);
			break;
		
		case EXPMSG_RAT_CONSUMED_FOOD:
			PLUGIN_RatConsumedFood();
			break;
			
		default:
			sprintf(buf, "%d", (int) msg->what);
			Debug('e', "ExperimenterPlugin::Start: Unexpected message received= ", buf);
			break;
		}
		
		delete msg;
	}
	
	return 0;
}
	 
BMessage *ExperimenterPlugin::GetSettings()
{
	BMessage *tmp = NULL;
	
	if (settings != NULL) {
	  	tmp = new BMessage(*settings);
	}
	
	return tmp;
}

void ExperimenterPlugin::EXP_Wait(float secs)
{
	snooze((int)(secs*1000000.0)); // BeOS specific call
}

/*
void ExperimenterPlugin::FlushIncomingMessages()
{
	envExpPort->Flush();
}	

BMessage *ExperimenterPlugin::GetIncomingMessage()
{
	return envExpPort->GetMessage();
}	

bool ExperimenterPlugin::IsIncomingMessageReady()
{
	return envExpPort->IsMessageReady();
}
*/

void ExperimenterPlugin::EXP_PutRatOnMaze()
{
	expRatPort->SendMessage(RATMSG_GO);
}

void ExperimenterPlugin::EXP_PutRatOnMaze(int trialType)
{
	BMessage msg;
	msg.what = RATMSG_GO;
	msg.AddInt16("trialType", (int16) trialType);
	expRatPort->SendMessage(&msg);
}

void ExperimenterPlugin::EXP_TakeRatOffMaze(float duration)
{
	BMessage msg;
	msg.what = RATMSG_STOP;
	msg.AddFloat("duration", duration);
	expRatPort->SendMessage(&msg);
}

void ExperimenterPlugin::EXP_SingleStepRat()
{
	expRatPort->SendMessage(RATMSG_SINGLE_STEP);
}

void ExperimenterPlugin::EXP_PlaceFood(int arm)
{
	userEnvMsg->PlaceFood(arm);
}

void ExperimenterPlugin::EXP_RemoveFood(int arm)
{
	userEnvMsg->RemoveFood(arm);
}

void ExperimenterPlugin::EXP_OpenDoor(int arm)
{
	userEnvMsg->OpenDoor(arm);
}

void ExperimenterPlugin::EXP_CloseDoor(int arm)
{
	userEnvMsg->CloseDoor(arm);
}

bool ExperimenterPlugin::EXP_AnyFoodOnMaze()
{
	bool isFood = false;
	
	int numArms = EXP_GetNumberOfMazeArms();
	
	for (int arm=0; arm < numArms; arm++) {
		if (EXP_IsFoodAtLocation(arm)) {
			isFood = true;
			break;
		}
	}
	
	return isFood;
}

bool ExperimenterPlugin::EXP_IsFoodAtLocation(int arm)
{
	return userEnvMsg->IsFoodAtLocation(arm);
}

int ExperimenterPlugin::EXP_GetRatHeading()
{
	return userEnvMsg->GetCurrHeading();
}

void ExperimenterPlugin::EXP_PlaceRatAt(int arm)
{
	userEnvMsg->WarpTo(arm);
}

int ExperimenterPlugin::EXP_GetNumberOfMazeArms()
{
	return userEnvMsg->GetNumberOfMazeArms();
}

int ExperimenterPlugin::EXP_GetNumberOfRats()
{
	return dbMsg->GetNumberOfRats();
}

int ExperimenterPlugin::EXP_GetCurrentRatNumber()
{
	return dbMsg->GetCurrentRatNumber();
}

void ExperimenterPlugin::EXP_SetCurrentRatNumber(int ratNo)
{
	dbMsg->SetCurrentRatNumber(ratNo);
}

void ExperimenterPlugin::EXP_AddDatabaseRecord(BMessage *record)
{
	dbMsg->AddRecord(record);
}

void ExperimenterPlugin::EXP_UserMessage(char *msg)
{
	DebugGUI(msg);
}

/*
BMessage *ExperimenterPlugin::WaitForMessage(unsigned int type)
{
	for (;;) {
		BMessage *bmsg = queue->Remove(); // block waiting for message
		if (bmsg->what == type) {
			return bmsg;
		} else {
			Debug('n', "ExperimenterPlugin::WaitForMessage: Received other message while",
				" waiting for rat");
		}
		delete bmsg;
	}
}
*/

int ExperimenterPlugin::EXP_WaitForEvent(int eventMask)
{
	int16 armNumber = -999;
	bool gotEvent = false;
	bool wrongEvent = false;
	
	while (! gotEvent) {
		BMessage *bmsg = queue->Remove(); // block waiting for message
		
		switch (bmsg->what) {
		case EXPMSG_RAT_MOVED:
			if (bmsg->FindInt16("armNumber", &armNumber) != B_OK) {
				Debug('e', "ExperimenterPlugin::EXP_WaitForEvent: No arm number in message");
			}
			if  (armNumber == CENTER) {
				if (eventMask & EXP_EVENT_CENTER_ENTRY) {
					gotEvent = true;
				} else {
					wrongEvent = true;
				}
			} else { // not center entry
				if (eventMask & EXP_EVENT_ARM_ENTRY) {
					gotEvent = true;
				} else {
					wrongEvent = true;
				}				
			}
			break;
			
		case EXPMSG_RAT_CONSUMED_FOOD:
			if (eventMask & EXP_EVENT_FOOD_CONSUMED) {
				gotEvent = true;
			} else {
				wrongEvent = true;
			}
			break;
			
		default:
			Debug('e', "ExperimenterPlugin::EXP_WaitForEvent: Received unexpected event!");
			break;
		}
		
		delete bmsg;
		
		if (wrongEvent) {
			Debug('n', "ExperimenterPlugin::EXP_WaitForEvent: Received other message while",
				" waiting for rat");
			wrongEvent = false;
		}
	}
	
	return (int) armNumber;
}

void ExperimenterPlugin::EXP_SaveSettings(BMessage *msg)
{
	int16 numberArms;
	
	// Change settings in database
	dbMsg->SetExperimenterSettings(settings);
	
	// Change number of arms with environment
	if (msg->FindInt16(NUMBER_OF_MAZE_ARMS, &numberArms) == B_OK) {
		userEnvMsg->SetNumberOfMazeArms((int) numberArms);
	}
}

// ExperimenterPlugin.cpp ends